/*
 * Decompiled with CFR 0.152.
 */
package com.bencodez.votingplugin.advancedcore.api.bedrock;

import com.bencodez.votingplugin.advancedcore.AdvancedCorePlugin;
import com.bencodez.votingplugin.advancedcore.api.user.AdvancedCoreUser;
import com.bencodez.votingplugin.advancedcore.api.user.UserManager;
import com.bencodez.votingplugin.advancedcore.api.user.UserStartup;
import java.lang.reflect.Method;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import org.bukkit.Bukkit;
import org.bukkit.entity.Player;

public final class BedrockNameResolver {
    private final BedrockDetect bedrockDetect;
    private final UserManager userManager;
    private final String bedrockPrefix;
    private final AdvancedCorePlugin plugin;
    private final Map<String, Boolean> cache = new ConcurrentHashMap<String, Boolean>();
    private final Map<String, String> ciIndex = new ConcurrentHashMap<String, String>();

    public BedrockNameResolver(final AdvancedCorePlugin plugin) {
        this.plugin = plugin;
        this.bedrockDetect = new BedrockDetect(plugin::debug);
        this.bedrockDetect.load();
        plugin.debug("[BedrockNameResolver] Floodgate loaded=" + this.bedrockDetect.isFloodgateAvailable() + ", Geyser loaded=" + this.bedrockDetect.isGeyserAvailable());
        this.userManager = plugin.getUserManager();
        this.bedrockPrefix = plugin.getOptions().getBedrockPlayerPrefix();
        plugin.addUserStartup(new UserStartup(){

            @Override
            public void onStart() {
                BedrockNameResolver.this.clearCache();
                plugin.debug("[BedrockNameResolver] startup: cleared in-memory cache/index");
            }

            @Override
            public void onStartUp(AdvancedCoreUser user) {
                BedrockNameResolver.this.learn(user);
            }

            @Override
            public void onFinish() {
                plugin.debug("BedrockNameResolver: startup loading complete");
            }
        });
    }

    public boolean isBedrock(String name) {
        return this.isBedrockName(name);
    }

    public boolean isBedrock(UUID uuid, String name) {
        if (uuid != null) {
            try {
                boolean viaUuid = this.bedrockDetect.isBedrock(uuid);
                if (viaUuid) {
                    this.plugin.debug("[BedrockNameResolver] isBedrock(uuid,name): TRUE via UUID");
                    return true;
                }
            }
            catch (Throwable viaUuid) {
                // empty catch block
            }
        }
        if (name != null && !name.isEmpty()) {
            for (Player p : Bukkit.getOnlinePlayers()) {
                if (!p.getName().equalsIgnoreCase(name)) continue;
                boolean online = this.bedrockDetect.isBedrock(p.getUniqueId());
                if (online) {
                    this.plugin.debug("[BedrockNameResolver] isBedrock(uuid,name): TRUE via online UUID match");
                    return true;
                }
                this.plugin.debug("[BedrockNameResolver] isBedrock(uuid,name): FALSE via online UUID match");
                return false;
            }
            Boolean cached = this.getCachedCaseInsensitive(name);
            if (cached != null) {
                this.plugin.debug("[BedrockNameResolver] isBedrock(uuid,name): " + cached + " via cache");
                return cached;
            }
            try {
                String canonical;
                AdvancedCoreUser user = this.userManager.getUser(name);
                if (user == null && (canonical = this.ciIndex.get(name.toLowerCase(Locale.ROOT))) != null) {
                    user = this.userManager.getUser(canonical);
                }
                if (user != null && user.isBedrockUser()) {
                    this.plugin.debug("[BedrockNameResolver] isBedrock(uuid,name): TRUE via DB flag");
                    return true;
                }
            }
            catch (Throwable throwable) {
                // empty catch block
            }
            if (this.bedrockPrefix != null && !this.bedrockPrefix.isEmpty() && name.startsWith(this.bedrockPrefix)) {
                this.plugin.debug("[BedrockNameResolver] isBedrock(uuid,name): TRUE via prefix fallback");
                return true;
            }
        }
        this.plugin.debug("[BedrockNameResolver] isBedrock(uuid,name): FALSE (no signals)");
        return false;
    }

    public void learn(AdvancedCoreUser user) {
        if (user == null) {
            return;
        }
        String name = user.getPlayerName();
        if (name == null || name.isEmpty()) {
            return;
        }
        this.putLearned(name, user.isBedrockUser());
    }

    public void learn(Player player) {
        if (player == null) {
            return;
        }
        UUID uuid = player.getUniqueId();
        boolean isBedrock = this.bedrockDetect.isBedrock(uuid);
        String originalName = player.getName();
        if (isBedrock) {
            this.plugin.debug("Learned Bedrock player: " + originalName + " (" + uuid + ")");
            this.cache.put(originalName, true);
            this.ciIndex.put(originalName.toLowerCase(Locale.ROOT), originalName);
            AdvancedCoreUser user = this.userManager.getUser(player);
            if (user != null) {
                user.setBedrockUser(true);
            }
        }
    }

    public boolean isBedrockName(String name) {
        if (name == null || name.isEmpty()) {
            return false;
        }
        for (Player p : Bukkit.getOnlinePlayers()) {
            if (!p.getName().equalsIgnoreCase(name)) continue;
            return this.bedrockDetect.isBedrock(p.getUniqueId());
        }
        Boolean cached = this.getCachedCaseInsensitive(name);
        if (cached != null) {
            return cached;
        }
        try {
            String canonical;
            AdvancedCoreUser user = this.userManager.getUser(name);
            if (user == null && (canonical = this.ciIndex.get(name.toLowerCase(Locale.ROOT))) != null) {
                user = this.userManager.getUser(canonical);
            }
            if (user != null && user.isBedrockUser()) {
                return true;
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        return this.bedrockPrefix != null && !this.bedrockPrefix.isEmpty() && name.startsWith(this.bedrockPrefix);
    }

    public Result resolve(String incomingName) {
        if (incomingName == null || incomingName.isEmpty()) {
            return new Result(incomingName, false, "empty-name");
        }
        for (Player p : Bukkit.getOnlinePlayers()) {
            if (!p.getName().equalsIgnoreCase(incomingName)) continue;
            boolean bedrock = this.bedrockDetect.isBedrock(p.getUniqueId());
            String finalName = this.addPrefixIfNeeded(incomingName, bedrock);
            return new Result(finalName, bedrock, bedrock ? "online-uuid-bedrock" : "online-uuid-java");
        }
        Boolean cached = this.getCachedCaseInsensitive(incomingName);
        if (cached != null) {
            boolean bedrock = cached;
            String finalName = this.addPrefixIfNeeded(incomingName, bedrock);
            return new Result(finalName, bedrock, "cache-" + (bedrock ? "bedrock" : "java"));
        }
        try {
            String canonical;
            AdvancedCoreUser user = this.userManager.getUser(incomingName);
            if (user == null && (canonical = this.ciIndex.get(incomingName.toLowerCase(Locale.ROOT))) != null) {
                user = this.userManager.getUser(canonical);
            }
            if (user != null) {
                boolean bedrock = user.isBedrockUser();
                String finalName = this.addPrefixIfNeeded(incomingName, bedrock);
                return new Result(finalName, bedrock, "db-" + (bedrock ? "bedrock" : "java"));
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        if (this.bedrockPrefix != null && !this.bedrockPrefix.isEmpty() && incomingName.startsWith(this.bedrockPrefix)) {
            return new Result(incomingName, true, "prefixed");
        }
        return new Result(incomingName, false, "unknown-default-java");
    }

    public String getPrefixedIfBedrock(String name) {
        return this.addPrefixIfNeeded(name, this.isBedrockName(name));
    }

    public void clearCache() {
        this.cache.clear();
        this.ciIndex.clear();
    }

    private void putLearned(String originalCaseName, boolean isBedrock) {
        if (originalCaseName == null || originalCaseName.isEmpty()) {
            return;
        }
        this.cache.put(originalCaseName, isBedrock);
        this.ciIndex.put(originalCaseName.toLowerCase(Locale.ROOT), originalCaseName);
    }

    private Boolean getCachedCaseInsensitive(String name) {
        Boolean exact = this.cache.get(name);
        if (exact != null) {
            return exact;
        }
        String canonical = this.ciIndex.get(name.toLowerCase(Locale.ROOT));
        return canonical != null ? this.cache.get(canonical) : null;
    }

    private String addPrefixIfNeeded(String name, boolean bedrock) {
        if (!bedrock) {
            return name;
        }
        if (this.bedrockPrefix == null || this.bedrockPrefix.isEmpty()) {
            return name;
        }
        if (name.startsWith(this.bedrockPrefix)) {
            return name;
        }
        return this.bedrockPrefix + name;
    }

    public static final class BedrockDetect {
        private volatile boolean floodgateAvailable = false;
        private volatile boolean geyserAvailable = false;
        private Object floodgateApi;
        private Method fgIsFloodgatePlayer;
        private Method fgGetPlayer;
        private Object geyserApi;
        private Method gzIsBedrockPlayer;
        private final Consumer<String> debug;

        public BedrockDetect() {
            this(s -> {});
        }

        public BedrockDetect(Consumer<String> debug) {
            this.debug = debug != null ? debug : s -> {};
        }

        public void load() {
            this.loadFloodgate();
            this.loadGeyser();
            this.debug.accept("[BedrockDetect] Loaded. Floodgate=" + this.floodgateAvailable + ", Geyser=" + this.geyserAvailable);
        }

        private void loadFloodgate() {
            try {
                Class<?> api = Class.forName("org.geysermc.floodgate.api.FloodgateApi");
                Method getInstance = api.getMethod("getInstance", new Class[0]);
                this.floodgateApi = getInstance.invoke(null, new Object[0]);
                this.fgIsFloodgatePlayer = this.floodgateApi.getClass().getMethod("isFloodgatePlayer", UUID.class);
                try {
                    this.fgGetPlayer = this.floodgateApi.getClass().getMethod("getPlayer", UUID.class);
                }
                catch (NoSuchMethodException noSuchMethodException) {
                    // empty catch block
                }
                this.floodgateAvailable = true;
                this.debug.accept("[BedrockDetect] Floodgate API: LOADED");
            }
            catch (Throwable t) {
                this.floodgateAvailable = false;
                this.debug.accept("[BedrockDetect] Floodgate API: NOT FOUND (" + t.getClass().getSimpleName() + ": " + t.getMessage() + ")");
            }
        }

        private void loadGeyser() {
            try {
                Class<?> apiClass = Class.forName("org.geysermc.geyser.api.GeyserApi");
                Method apiMethod = apiClass.getMethod("api", new Class[0]);
                this.geyserApi = apiMethod.invoke(null, new Object[0]);
                this.gzIsBedrockPlayer = this.geyserApi.getClass().getMethod("isBedrockPlayer", UUID.class);
                this.geyserAvailable = true;
                this.debug.accept("[BedrockDetect] Geyser API: LOADED");
            }
            catch (Throwable t) {
                this.geyserAvailable = false;
                this.debug.accept("[BedrockDetect] Geyser API: NOT FOUND (" + t.getClass().getSimpleName() + ": " + t.getMessage() + ")");
            }
        }

        public boolean isBedrock(UUID uuid) {
            Object v;
            if (uuid == null) {
                return false;
            }
            if (this.floodgateAvailable) {
                try {
                    v = this.fgIsFloodgatePlayer.invoke(this.floodgateApi, uuid);
                    if (v instanceof Boolean && ((Boolean)v).booleanValue()) {
                        return true;
                    }
                }
                catch (Throwable t) {
                    this.floodgateAvailable = false;
                    this.debug.accept("[BedrockDetect] Floodgate call failed, disabling: " + t.getClass().getSimpleName());
                }
            }
            if (this.geyserAvailable) {
                try {
                    v = this.gzIsBedrockPlayer.invoke(this.geyserApi, uuid);
                    if (v instanceof Boolean) {
                        return (Boolean)v;
                    }
                }
                catch (Throwable t) {
                    this.geyserAvailable = false;
                    this.debug.accept("[BedrockDetect] Geyser call failed, disabling: " + t.getClass().getSimpleName());
                }
            }
            return false;
        }

        public Object getFloodgatePlayer(UUID uuid) {
            if (!this.floodgateAvailable || this.fgGetPlayer == null || uuid == null) {
                return null;
            }
            try {
                return this.fgGetPlayer.invoke(this.floodgateApi, uuid);
            }
            catch (Throwable t) {
                this.floodgateAvailable = false;
                return null;
            }
        }

        public boolean isFloodgateAvailable() {
            return this.floodgateAvailable;
        }

        public boolean isGeyserAvailable() {
            return this.geyserAvailable;
        }
    }

    public static final class Result {
        public final String finalName;
        public final boolean isBedrock;
        public final String rationale;

        public Result(String finalName, boolean isBedrock, String rationale) {
            this.finalName = finalName;
            this.isBedrock = isBedrock;
            this.rationale = rationale;
        }
    }
}

