/*
 * Decompiled with CFR 0.152.
 */
package org.mvplugins.multiverse.inventories.profile;

import com.google.common.base.Strings;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.FileConfiguration;
import org.jetbrains.annotations.NotNull;
import org.jvnet.hk2.annotations.Service;
import org.mvplugins.multiverse.core.exceptions.MultiverseException;
import org.mvplugins.multiverse.external.jakarta.inject.Inject;
import org.mvplugins.multiverse.external.vavr.CheckedFunction0;
import org.mvplugins.multiverse.external.vavr.control.Option;
import org.mvplugins.multiverse.external.vavr.control.Try;
import org.mvplugins.multiverse.inventories.profile.AsyncFileIO;
import org.mvplugins.multiverse.inventories.profile.GlobalProfile;
import org.mvplugins.multiverse.inventories.profile.PlayerNamesMapper;
import org.mvplugins.multiverse.inventories.profile.PlayerProfileJsonSerializer;
import org.mvplugins.multiverse.inventories.profile.ProfileCacheManager;
import org.mvplugins.multiverse.inventories.profile.ProfileDataSource;
import org.mvplugins.multiverse.inventories.profile.ProfileFilesLocator;
import org.mvplugins.multiverse.inventories.profile.data.PlayerProfile;
import org.mvplugins.multiverse.inventories.profile.key.ContainerType;
import org.mvplugins.multiverse.inventories.profile.key.GlobalProfileKey;
import org.mvplugins.multiverse.inventories.profile.key.ProfileFileKey;
import org.mvplugins.multiverse.inventories.profile.key.ProfileKey;
import org.mvplugins.multiverse.inventories.profile.key.ProfileType;
import org.mvplugins.multiverse.inventories.profile.key.ProfileTypes;
import org.mvplugins.multiverse.inventories.utils.InvLogging;
import org.mvplugins.multiverse.inventories.utils.configuration.json.JsonConfiguration;

@Service
final class FlatFileProfileDataSource
implements ProfileDataSource {
    private final AsyncFileIO asyncFileIO;
    private final ProfileFilesLocator profileFilesLocator;
    private final ProfileCacheManager profileCacheManager;
    private final PlayerNamesMapper playerNamesMapper;

    @Inject
    FlatFileProfileDataSource(@NotNull AsyncFileIO asyncFileIO, @NotNull ProfileFilesLocator profileFilesLocator, @NotNull ProfileCacheManager profileCacheManager, @NotNull PlayerNamesMapper playerNamesMapper) {
        this.asyncFileIO = asyncFileIO;
        this.profileFilesLocator = profileFilesLocator;
        this.profileCacheManager = profileCacheManager;
        this.playerNamesMapper = playerNamesMapper;
    }

    private FileConfiguration loadFileToJsonConfiguration(File file) {
        JsonConfiguration jsonConfiguration = new JsonConfiguration();
        jsonConfiguration.options().continueOnSerializationError(false);
        Try.run(() -> jsonConfiguration.load(file)).getOrElseThrow(e -> {
            InvLogging.severe("Could not load file %s : %s", file, e.getMessage());
            e.printStackTrace();
            throw new RuntimeException((Throwable)e);
        });
        return jsonConfiguration;
    }

    private FileConfiguration getOrLoadPlayerProfileFile(ProfileFileKey profileKey, File playerFile) {
        return (FileConfiguration)Try.of((CheckedFunction0 & Serializable)() -> this.profileCacheManager.getOrLoadPlayerFile(profileKey, key -> playerFile.exists() ? this.loadFileToJsonConfiguration(playerFile) : new JsonConfiguration())).getOrElseThrow(e -> {
            InvLogging.severe("Could not load profile data for player: " + String.valueOf(profileKey), new Object[0]);
            return new RuntimeException((Throwable)e);
        });
    }

    @Override
    public CompletableFuture<PlayerProfile> getPlayerProfile(ProfileKey profileKey) {
        try {
            if (Strings.isNullOrEmpty((String)profileKey.getPlayerName())) {
                return CompletableFuture.failedFuture(new IllegalArgumentException("Player name cannot be null or empty. " + String.valueOf(profileKey)));
            }
            return this.profileCacheManager.getOrLoadPlayerProfile(profileKey, (key, executor) -> {
                File playerFile = this.profileFilesLocator.getPlayerProfileFile(profileKey);
                if (!playerFile.exists()) {
                    InvLogging.fine("Not found on disk: %s", playerFile);
                    return CompletableFuture.completedFuture(PlayerProfile.newProfile(key));
                }
                InvLogging.finer("%s not cached. loading from disk...", profileKey);
                return this.asyncFileIO.queueFileCallable(playerFile, () -> this.getPlayerProfileFromDisk((ProfileKey)key, playerFile));
            });
        }
        catch (Exception e) {
            InvLogging.severe("Could not get data for player: " + profileKey.getPlayerName() + " for " + profileKey.getContainerType().toString() + ": " + profileKey.getDataName(), new Object[0]);
            throw new RuntimeException(e);
        }
    }

    private PlayerProfile getPlayerProfileFromDisk(ProfileKey key, File playerFile) {
        FileConfiguration playerData = this.getOrLoadPlayerProfileFile(key, playerFile);
        ConfigurationSection section = playerData.getConfigurationSection(key.getProfileType().getName());
        if (section == null || section.getKeys(false).isEmpty()) {
            return PlayerProfile.newProfile(key);
        }
        return PlayerProfileJsonSerializer.deserialize(key, this.convertSection(section));
    }

    private Map<String, Object> convertSection(ConfigurationSection section) {
        Set keys = section.getKeys(false);
        HashMap<String, Object> resultMap = new HashMap<String, Object>(keys.size());
        for (String key : keys) {
            Object obj = section.get(key);
            if (obj instanceof ConfigurationSection) {
                resultMap.put(key, this.convertSection((ConfigurationSection)obj));
                continue;
            }
            resultMap.put(key, obj);
        }
        return resultMap;
    }

    @Override
    public CompletableFuture<Void> updatePlayerProfile(PlayerProfile playerProfile) {
        ProfileKey profileKey = ProfileKey.fromPlayerProfile(playerProfile);
        File playerFile = this.profileFilesLocator.getPlayerProfileFile(profileKey);
        return this.asyncFileIO.queueFileAction(playerFile, () -> this.savePlayerProfileToDisk(profileKey, playerFile, playerProfile.clone()));
    }

    private void savePlayerProfileToDisk(ProfileKey profileKey, File playerFile, PlayerProfile playerProfile) {
        FileConfiguration playerData = this.getOrLoadPlayerProfileFile(profileKey, playerFile);
        Map<String, Object> serializedData = PlayerProfileJsonSerializer.serialize(playerProfile);
        if (serializedData.isEmpty()) {
            return;
        }
        playerData.createSection(playerProfile.getProfileType().getName(), serializedData);
        Try.run(() -> playerData.save(playerFile)).onFailure(e -> {
            InvLogging.severe("Could not save data for player: " + playerProfile.getPlayerName() + " for " + String.valueOf((Object)playerProfile.getContainerType()) + ": " + playerProfile.getContainerName(), new Object[0]);
            e.printStackTrace();
        });
    }

    @Override
    public CompletableFuture<Void> deletePlayerProfile(ProfileKey profileKey) {
        if (Strings.isNullOrEmpty((String)profileKey.getPlayerName())) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("Player name cannot be null or empty. " + String.valueOf(profileKey)));
        }
        File playerFile = this.profileFilesLocator.getPlayerProfileFile(profileKey);
        this.profileCacheManager.getCachedPlayerProfile(profileKey).peek(profile -> profile.getData().clear());
        return this.asyncFileIO.queueFileAction(playerFile, () -> this.deletePlayerProfileFromDisk(profileKey, playerFile, new ProfileType[]{profileKey.getProfileType()}));
    }

    @Override
    public CompletableFuture<Void> deletePlayerProfiles(ProfileFileKey profileKey, ProfileType[] profileTypes) {
        if (Strings.isNullOrEmpty((String)profileKey.getPlayerName())) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("Player name cannot be null or empty. " + String.valueOf(profileKey)));
        }
        if (ProfileTypes.isAll(profileTypes)) {
            InvLogging.finer("Deleting profile: " + String.valueOf(profileKey) + " for all profile-types", new Object[0]);
            return this.deletePlayerFile(profileKey);
        }
        for (ProfileType profileType : profileTypes) {
            this.profileCacheManager.getCachedPlayerProfile(profileKey.forProfileType(profileType)).peek(profile -> profile.getData().clear());
        }
        File playerFile = this.profileFilesLocator.getPlayerProfileFile(profileKey);
        return this.asyncFileIO.queueFileAction(playerFile, () -> this.deletePlayerProfileFromDisk(profileKey, playerFile, profileTypes));
    }

    private void deletePlayerProfileFromDisk(ProfileFileKey profileKey, File playerFile, ProfileType[] profileTypes) {
        try {
            FileConfiguration playerData = this.getOrLoadPlayerProfileFile(profileKey, playerFile);
            for (ProfileType profileType : profileTypes) {
                playerData.set(profileType.getName(), null);
            }
            playerData.save(playerFile);
        }
        catch (IOException e) {
            InvLogging.severe("Could not delete data for player: " + profileKey.getPlayerName() + " for " + String.valueOf((Object)profileKey.getContainerType()) + ": " + profileKey.getDataName(), new Object[0]);
            InvLogging.severe(e.getMessage(), new Object[0]);
        }
    }

    @Override
    public CompletableFuture<Void> deletePlayerFile(ProfileFileKey profileKey) {
        for (ProfileType type : ProfileTypes.getTypes()) {
            this.profileCacheManager.getCachedPlayerProfile(profileKey.forProfileType(type)).peek(profile -> profile.getData().clear());
        }
        File playerFile = this.profileFilesLocator.getPlayerProfileFile(profileKey);
        if (!playerFile.exists()) {
            InvLogging.finer("Attempted to delete file that did not exist for player " + profileKey.getPlayerName() + " in " + String.valueOf((Object)profileKey.getContainerType()) + " " + profileKey.getDataName(), new Object[0]);
            return CompletableFuture.completedFuture(null);
        }
        return this.asyncFileIO.queueFileAction(playerFile, () -> {
            if (!playerFile.delete()) {
                InvLogging.warning("Could not delete file for player " + profileKey.getPlayerName() + " in " + String.valueOf((Object)profileKey.getContainerType()) + " " + profileKey.getDataName(), new Object[0]);
            }
        });
    }

    @Override
    public CompletableFuture<Void> clonePlayerProfiles(ProfileFileKey fromProfileKey, ProfileFileKey toProfileKey, ProfileType[] profileTypes) {
        if (Strings.isNullOrEmpty((String)fromProfileKey.getPlayerName())) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("Player name cannot be null or empty. " + String.valueOf(fromProfileKey)));
        }
        if (Strings.isNullOrEmpty((String)toProfileKey.getPlayerName())) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("Player name cannot be null or empty. " + String.valueOf(toProfileKey)));
        }
        if (ProfileTypes.isAll(profileTypes)) {
            InvLogging.finer("Clone profile from " + String.valueOf(fromProfileKey) + " to " + String.valueOf(toProfileKey) + " for all profile-types", new Object[0]);
            return this.clonePlayerFile(fromProfileKey, toProfileKey);
        }
        this.profileCacheManager.clearCacheForFile(toProfileKey);
        File fromPlayerFile = this.profileFilesLocator.getPlayerProfileFile(fromProfileKey);
        File toPlayerFile = this.profileFilesLocator.getPlayerProfileFile(toProfileKey);
        return this.asyncFileIO.queueFilesAction(new File[]{fromPlayerFile, toPlayerFile}, () -> this.cloneSpecificProfiles(fromProfileKey, fromPlayerFile, toProfileKey, toPlayerFile, profileTypes));
    }

    private void cloneSpecificProfiles(ProfileFileKey fromProfileKey, File fromPlayerFile, ProfileFileKey toProfileKey, File toPlayerFile, ProfileType[] profileTypes) {
        FileConfiguration fromPlayerData = this.getOrLoadPlayerProfileFile(fromProfileKey, fromPlayerFile);
        FileConfiguration toPlayerData = this.getOrLoadPlayerProfileFile(toProfileKey, toPlayerFile);
        for (ProfileType profileType : profileTypes) {
            ConfigurationSection fromSection = fromPlayerData.getConfigurationSection(profileType.getName());
            if (fromSection != null && !fromSection.getKeys(false).isEmpty()) {
                toPlayerData.set(profileType.getName(), (Object)fromSection);
                continue;
            }
            toPlayerData.set(profileType.getName(), null);
        }
        Try.run(() -> toPlayerData.save(toPlayerFile)).onFailure(e -> {
            InvLogging.severe("Could not clone specific profiles from player " + fromProfileKey.getPlayerName() + " to " + toProfileKey.getPlayerName() + " in " + String.valueOf((Object)fromProfileKey.getContainerType()) + " " + fromProfileKey.getDataName(), new Object[0]);
            InvLogging.severe(e.getMessage(), new Object[0]);
        });
    }

    @Override
    public CompletableFuture<Void> clonePlayerFile(ProfileFileKey fromProfileKey, ProfileFileKey toProfileKey) {
        File fromPlayerFile = this.profileFilesLocator.getPlayerProfileFile(fromProfileKey);
        File toPlayerFile = this.profileFilesLocator.getPlayerProfileFile(toProfileKey);
        if (!fromPlayerFile.exists()) {
            InvLogging.finer("Attempted to clone file that did not exist for player " + fromProfileKey.getPlayerName() + " in " + String.valueOf((Object)fromProfileKey.getContainerType()) + " " + fromProfileKey.getDataName(), new Object[0]);
            return CompletableFuture.completedFuture(null);
        }
        this.profileCacheManager.clearCacheForFile(toProfileKey);
        return this.asyncFileIO.queueFilesAction(new File[]{fromPlayerFile, toPlayerFile}, () -> Try.run(() -> Files.copy(fromPlayerFile.toPath(), toPlayerFile.toPath(), StandardCopyOption.REPLACE_EXISTING)).onFailure(e -> {
            InvLogging.severe("Could not clone file " + String.valueOf(fromPlayerFile) + " to " + String.valueOf(toPlayerFile), new Object[0]);
            e.printStackTrace();
        }));
    }

    @Override
    public void migratePlayerProfileName(String oldName, String newName) {
        this.profileCacheManager.clearPlayerCache(oldName);
        List<File> worldFolders = this.profileFilesLocator.listProfileContainerFolders(ContainerType.WORLD);
        List<File> groupFolders = this.profileFilesLocator.listProfileContainerFolders(ContainerType.GROUP);
        this.migrateForContainerType(worldFolders, ContainerType.WORLD, oldName, newName);
        this.migrateForContainerType(groupFolders, ContainerType.GROUP, oldName, newName);
    }

    private void migrateForContainerType(List<File> folders, ContainerType containerType, String oldName, String newName) {
        for (File folder : folders) {
            File oldNameFile = this.profileFilesLocator.getPlayerProfileFile(containerType, folder.getName(), oldName);
            File newNameFile = this.profileFilesLocator.getPlayerProfileFile(containerType, folder.getName(), newName);
            if (!oldNameFile.exists()) {
                InvLogging.fine("No old data for player %s in %s %s to migrate.", oldName, containerType.name(), folder.getName());
                continue;
            }
            if (newNameFile.exists()) {
                InvLogging.warning("Data already exists for player %s in %s %s. Not migrating.", newName, containerType.name(), folder.getName());
                continue;
            }
            if (!oldNameFile.renameTo(newNameFile)) {
                InvLogging.warning("Could not rename old data file for player %s in %s %s to %s.", oldName, containerType.name(), folder.getName(), newName);
                continue;
            }
            InvLogging.fine("Migrated data for player %s in %s %s to %s.", oldName, containerType.name(), folder.getName(), newName);
        }
    }

    @Override
    public CompletableFuture<GlobalProfile> getGlobalProfile(GlobalProfileKey key) {
        File globalFile = this.profileFilesLocator.getGlobalFile(key.getPlayerUUID());
        return this.profileCacheManager.getOrLoadGlobalProfile(key.getPlayerUUID(), (uuid, executor) -> {
            InvLogging.finer("Global profile for player %s (%s) not in cached. Loading...", uuid, key.getPlayerName());
            this.migrateGlobalProfileToUUID((UUID)uuid, key.getPlayerName());
            if (!globalFile.exists()) {
                GlobalProfile globalProfile = new GlobalProfile(key.getPlayerUUID(), globalFile.toPath());
                globalProfile.setLastKnownName(key.getPlayerName());
                return CompletableFuture.completedFuture(globalProfile);
            }
            return this.asyncFileIO.queueFileCallable(globalFile, () -> this.getGlobalProfileFromDisk(key.getPlayerUUID(), globalFile));
        });
    }

    @Override
    public CompletableFuture<Option<GlobalProfile>> getExistingGlobalProfile(GlobalProfileKey key) {
        File uuidFile = this.profileFilesLocator.getGlobalFile(key.getPlayerUUID());
        if (!uuidFile.exists()) {
            return CompletableFuture.completedFuture(Option.none());
        }
        return this.getGlobalProfile(key).thenApply(Option::of);
    }

    private void migrateGlobalProfileToUUID(UUID playerUUID, String playerName) {
        File legacyFile = this.profileFilesLocator.getGlobalFile(playerName);
        if (!legacyFile.exists()) {
            return;
        }
        if (!legacyFile.renameTo(this.profileFilesLocator.getGlobalFile(playerUUID.toString()))) {
            InvLogging.warning("Could not properly migrate player global data file for " + playerName, new Object[0]);
        }
    }

    private GlobalProfile getGlobalProfileFromDisk(UUID playerUUID, File globalFile) {
        return new GlobalProfile(playerUUID, globalFile.toPath());
    }

    @Override
    public CompletableFuture<Void> modifyGlobalProfile(GlobalProfileKey key, Consumer<GlobalProfile> consumer) {
        return this.getGlobalProfile(key).thenCompose(globalProfile -> this.modifyGlobalProfile((GlobalProfile)globalProfile, consumer));
    }

    private CompletableFuture<Void> modifyGlobalProfile(GlobalProfile globalProfile, Consumer<GlobalProfile> consumer) {
        consumer.accept(globalProfile);
        return this.updateGlobalProfile(globalProfile);
    }

    @Override
    public CompletableFuture<Void> updateGlobalProfile(GlobalProfile globalProfile) {
        File globalFile = this.profileFilesLocator.getGlobalFile(globalProfile.getPlayerUUID().toString());
        boolean didPlayerNameChange = this.playerNamesMapper.setPlayerName(globalProfile.getPlayerUUID(), globalProfile.getLastKnownName());
        return this.asyncFileIO.queueFileAction(globalFile, () -> this.processGlobalProfileWrite(globalProfile)).thenCompose(ignore -> didPlayerNameChange ? this.playerNamesMapper.savePlayerNames() : CompletableFuture.completedFuture(null));
    }

    private void processGlobalProfileWrite(GlobalProfile globalProfile) {
        globalProfile.save().onFailure(throwable -> {
            InvLogging.severe("Could not save global data for player: " + String.valueOf(globalProfile), new Object[0]);
            InvLogging.severe(throwable.getMessage(), new Object[0]);
        });
    }

    @Override
    public CompletableFuture<Void> deleteGlobalProfile(GlobalProfileKey key, boolean clearPlayerFiles) {
        return ((CompletableFuture)this.getExistingGlobalProfile(key).thenCompose(globalProfile -> {
            if (globalProfile.isEmpty()) {
                return CompletableFuture.failedFuture((Throwable)new MultiverseException("Invalid global profile for player: " + String.valueOf(key)));
            }
            return this.deleteGlobalProfileFromDisk((GlobalProfile)globalProfile.get());
        })).thenCompose(ignore -> clearPlayerFiles ? this.clearAllPlayerProfileFiles(key.getPlayerUUID(), key.getPlayerName()) : CompletableFuture.completedFuture(null));
    }

    private CompletableFuture<Void> deleteGlobalProfileFromDisk(GlobalProfile globalProfile) {
        File globalFile = this.profileFilesLocator.getGlobalFile(globalProfile.getPlayerUUID().toString());
        return this.asyncFileIO.queueFileAction(globalFile, () -> {
            if (!globalFile.delete()) {
                throw new RuntimeException("Could not delete global profile file: " + String.valueOf(globalFile));
            }
        });
    }

    private CompletableFuture<Void> clearAllPlayerProfileFiles(UUID playerUUID, String playerName) {
        return CompletableFuture.allOf((CompletableFuture[])Arrays.stream(ContainerType.values()).flatMap(containerType -> this.listContainerDataNames((ContainerType)((Object)containerType)).stream().map(containerName -> this.deletePlayerFile(ProfileFileKey.of(containerType, containerName, playerUUID, playerName)))).toArray(CompletableFuture[]::new));
    }

    @Override
    public List<String> listContainerDataNames(ContainerType containerType) {
        return this.profileFilesLocator.listProfileContainerFolders(containerType).stream().map(File::getName).toList();
    }

    @Override
    public List<String> listPlayerProfileNames(ContainerType containerType, String containerName) {
        return this.profileFilesLocator.listPlayerProfileFiles(containerType, containerName).stream().map(file -> com.google.common.io.Files.getNameWithoutExtension((String)file.getName())).toList();
    }

    @Override
    public List<UUID> listGlobalProfileUUIDs() {
        return this.profileFilesLocator.listGlobalFiles().stream().map(file -> {
            String fileName = com.google.common.io.Files.getNameWithoutExtension((String)file.getName());
            return (UUID)Try.of((CheckedFunction0 & Serializable)() -> UUID.fromString(fileName)).onFailure(throwable -> {
                InvLogging.warning("File name is not a valid UUID: %s", fileName);
                InvLogging.warning(throwable.getMessage(), new Object[0]);
            }).getOrNull();
        }).filter(Objects::nonNull).toList();
    }
}

